<?php
declare(strict_types=1);
session_start();

/**
 * Instellingen
 */
const TEMPLATE_DIR = __DIR__ . '/def';            // bronmap die je wilt klonen
const DEST_ROOT    = __DIR__ . '/custom_map123';  // hier komen de kopieën
const COOKIE_NAME  = 'vo_bucket';
const COOKIE_TTL   = 86400 * 7;                   // 7 dagen hergebruik

/**
 * Hulpfuncties
 */
function ensure_dir(string $path): void {
    if (!is_dir($path) && !mkdir($path, 0755, true) && !is_dir($path)) {
        http_response_code(500);
        exit('Kon doelmap niet aanmaken.');
    }
}

function rr_copy(string $src, string $dst): void {
    if (!is_dir($src)) {
        http_response_code(500);
        exit('Bronmap ontbreekt: ' . htmlspecialchars($src));
    }
    ensure_dir($dst);

    $it = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($src, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ($it as $item) {
        $target = $dst . DIRECTORY_SEPARATOR . $it->getSubPathName();

        if ($item->isDir()) {
            if (!is_dir($target) && !mkdir($target, 0755, true) && !is_dir($target)) {
                http_response_code(500);
                exit('Kon submap niet aanmaken: ' . htmlspecialchars($target));
            }
        } else {
            // voorkom symlink-trucs; kopieer alleen gewone bestanden
            if ($item->isLink()) {
                continue;
            }
            if (!copy($item->getPathname(), $target)) {
                http_response_code(500);
                exit('Kopiëren mislukt: ' . htmlspecialchars($item->getPathname()));
            }
            // optioneel: permissies normaliseren
            @chmod($target, 0644);
        }
    }
}

/**
 * Bepaal of deze bezoeker al een bucket heeft
 */
$bucket = $_COOKIE[COOKIE_NAME] ?? '';

if ($bucket !== '' && preg_match('/^[a-f0-9]{16,64}$/i', $bucket)) {
    $destDir = DEST_ROOT . '/' . $bucket;
    if (!is_dir($destDir)) {
        // bestond niet meer; maak opnieuw
        $bucket = '';
    }
}

if ($bucket === '') {
    // Maak een nieuwe, cryptografisch sterke ID
    $bucket = bin2hex(random_bytes(8)); // 16 hex chars
    $destDir = DEST_ROOT . '/' . $bucket;

    // Maak de map en kopieer de template
    ensure_dir(DEST_ROOT);
    rr_copy(TEMPLATE_DIR, $destDir);

    // Zet cookie voor hergebruik
    $secure = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
    setcookie(
        COOKIE_NAME,
        $bucket,
        [
            'expires'  => time() + COOKIE_TTL,
            'path'     => '/vo',     // beperkt tot deze sectie
            'secure'   => $secure,   // alleen via HTTPS verzenden
            'httponly' => true,
            'samesite' => 'Lax',
        ]
    );
}

// Redirect naar de persoonlijke map, querystring meenemen
$q = $_SERVER['QUERY_STRING'] ?? '';
$targetRel = 'custom_map123/' . rawurlencode($bucket) . '/';
$location  = $targetRel . ($q ? ('?' . $q) : '');

header('Location: ' . $location, true, 302);
?>
<!doctype html>
<html lang="nl">
<head>
  <meta charset="utf-8">
  <title>Bezig met doorsturen…</title>
  <meta http-equiv="refresh" content="1;url=<?= htmlspecialchars($location, ENT_QUOTES) ?>">
  <style>body{font-family:system-ui,Segoe UI,Roboto,Arial,sans-serif;padding:2rem;color:#333}</style>
</head>
<body>
  <p>Even geduld… je wordt doorgestuurd.</p>
  <p><a href="<?= htmlspecialchars($location, ENT_QUOTES) ?>">Klik hier als het niet automatisch gaat.</a></p>
  <script>setTimeout(function(){location.href="<?= addslashes($location) ?>";}, 300);</script>
</body>
</html>
